/*
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.internal.tasks.testing.worker

import com.google.common.collect.ImmutableList
import org.gradle.api.Action
import org.gradle.api.internal.tasks.testing.TestDefinition
import org.gradle.api.internal.tasks.testing.WorkerTestClassProcessorFactory
import org.gradle.internal.exceptions.DefaultMultiCauseException
import org.gradle.internal.remote.ObjectConnection
import org.gradle.internal.work.WorkerThreadRegistry
import org.gradle.process.JavaForkOptions
import org.gradle.process.ProcessExecutionException
import org.gradle.process.internal.JavaExecHandleBuilder
import org.gradle.process.internal.worker.WorkerProcess
import org.gradle.process.internal.worker.WorkerProcessBuilder
import org.gradle.process.internal.worker.WorkerProcessFactory
import spock.lang.Specification

class ForkingTestClassProcessorTest extends Specification {
    WorkerThreadRegistry workerLeaseRegistry = Mock(WorkerThreadRegistry)
    RemoteTestClassProcessor remoteProcessor = Mock(RemoteTestClassProcessor)
    ObjectConnection connection = Mock(ObjectConnection) {
        addOutgoing(RemoteTestClassProcessor.class) >> remoteProcessor
    }
    WorkerProcess workerProcess = Mock(WorkerProcess) {
        getConnection() >> connection
    }
    WorkerProcessBuilder workerProcessBuilder = Mock(WorkerProcessBuilder) {
        build() >> workerProcess
        getJavaCommand() >> Stub(JavaExecHandleBuilder)
    }
    WorkerProcessFactory workerProcessFactory = Stub(WorkerProcessFactory) {
        create(_) >> workerProcessBuilder
    }

    def "acquires worker lease and starts worker process on first test"() {
        given:
        def test1 = Mock(TestDefinition)
        def test2 = Mock(TestDefinition)
        def processor = newProcessor()

        when:
        processor.processTestDefinition(test1)
        processor.processTestDefinition(test2)

        then:
        1 * workerLeaseRegistry.startWorker()
        1 * remoteProcessor.processTestDefinition(test1)
        1 * remoteProcessor.processTestDefinition(test2)
        1 * remoteProcessor.startProcessing()
        0 * remoteProcessor._
    }

    def "starts process with the specified classpath"() {
        given:
        def appClasspath = ImmutableList.of(new File("cls.jar"))
        def appModulepath = ImmutableList.of(new File("mod.jar"))
        def implClasspath = ImmutableList.of(new URL("file://cls.jar"))
        def processor = newProcessor(new ForkedTestClasspath(
            appClasspath, appModulepath, implClasspath
        ))

        when:
        processor.forkProcess()

        then:
        1 * workerProcessBuilder.applicationClasspath(_) >> { assert it[0] == appClasspath }
        1 * workerProcessBuilder.applicationModulePath(_) >> { assert it[0] == appModulepath}
        1 * workerProcessBuilder.setImplementationClasspath(_) >> { assert it[0] == implClasspath }
    }

    def "stopNow does nothing when no remote processor"() {
        given:
        def processor = newProcessor()

        when:
        processor.stopNow()

        then:
        0 * _
    }

    def "stopNow propagates to worker process"() {
        given:
        def processor = newProcessor()

        when:
        processor.processTestDefinition(Mock(TestDefinition))
        processor.stopNow()

        then:
        1 * workerProcess.stopNow()
    }

    def "no exception when stop after stopNow"() {
        def processor = newProcessor()

        when:
        processor.processTestDefinition(Mock(TestDefinition))
        processor.stopNow()
        processor.stop()

        then:
        1 * workerProcess.stopNow()
        _ * workerProcess.waitForStop() >> { throw new ProcessExecutionException("waitForStop can throw") }
        notThrown(ProcessExecutionException)
    }

    def "captures and rethrows unrecoverable exceptions thrown by the connection"() {
        def handler
        def processor = newProcessor()

        when:
        processor.processTestDefinition(Mock(TestDefinition))

        then:
        1 * workerProcess.getConnection() >> Stub(ObjectConnection) {
            addUnrecoverableErrorHandler(_) >> { args -> handler = args[0] }
            addOutgoing(_) >> Stub(RemoteTestClassProcessor)
        }

        when:
        def unexpectedException = new Throwable('BOOM!')
        handler.execute(unexpectedException)

        and:
        processor.stop()

        then:
        def e = thrown(DefaultMultiCauseException)
        e.causes.contains(unexpectedException)
    }

    def "ignores unrecoverable exceptions after stopNow() is called"() {
        def handler
        def processor = newProcessor()

        when:
        processor.processTestDefinition(Mock(TestDefinition))

        then:
        1 * workerProcess.getConnection() >> Stub(ObjectConnection) {
            addUnrecoverableErrorHandler(_) >> { args -> handler = args[0] }
            addOutgoing(_) >> Stub(RemoteTestClassProcessor)
        }

        when:
        processor.stopNow()
        def unexpectedException = new Throwable('BOOM!')
        handler.execute(unexpectedException)

        and:
        processor.stop()

        then:
        noExceptionThrown()
    }

    def newProcessor(
        ForkedTestClasspath classpath = new ForkedTestClasspath(ImmutableList.of(), ImmutableList.of(), ImmutableList.of())
    ) {
        return new ForkingTestClassProcessor(
            workerLeaseRegistry, workerProcessFactory, Mock(WorkerTestClassProcessorFactory),
            Stub(JavaForkOptions), classpath, Mock(Action)
        )
    }
}
